/************************************************************************/
/*    FDUPS.c	Program to scan a disk for duplicate files		*/
/*									*/
/* 		Author: W. A. Smethurst - 73657,2447 (CompuServe)	*/
/*		Date:	07/04/95					*/
/************************************************************************/

#include "stdio.h"
#include "stdlib.h"
#include "process.h"
#include "dos.h"
#include "conio.h"
#include "ctype.h"
#include "alloc.h"
#include "dir.h"
#include "errno.h"

/************************************************************************/
/* Global Definitions							*/
/************************************************************************/

#define SUCCESS		0			  /* Function succeeded	*/
#define FAIL		-1		       /* Function unsuccessful	*/
#define EXIT		1			 /* Exit Program switch	*/
#define CANCEL		27				  /* Escape key	*/
#define BKSLASH		92				    /* Hex 0x5c	*/
#define FALSE		0
#define TRUE		!FALSE

#ifndef   BOOLTYPE
  #define BOOLTYPE 0
  typedef unsigned short int BOOL;
#endif

typedef struct dfile		      /* File name structure definition	*/
	{
	char filename[13];
	unsigned ftime;
	unsigned fdate;
	unsigned long fsize;
	char *parent;		    /* Pointer to parent directory path	*/
	struct dfile far *next;
	} DFILE;

/************************************************************************/
/* Global Variable Declarations						*/
/************************************************************************/

char *ProgTitle = "FDUPS v1.0 - Find and Report Duplicate Files";

int drive;
int row,col;
int dircnt;
int filcnt;
int dupcnt;

DFILE far *froot;

char cdrive[4];		/* Root directory of designated drive	*/
char curdir[MAXDIR];	/* Current working directory	*/

/************************************************************************/
/* External String and Function Declarations				*/
/************************************************************************/

extern char *sys_errlist[];	  		  /* DOS Error Messages	*/

/************************************************************************/
/* Global Utility Function Declarations.               			*/
/************************************************************************/

char *EditDate(unsigned date);
char *EditTime(unsigned time);
void ClearRow(int row);
void SetCursor(int row,int col);
unsigned Row(),Col();

/************************************************************************/
/* FDUPS Main Program Function						*/
/************************************************************************/

main(int argc, char *argv[])
{
	int Instructions(),initialize(),MainLine();

	int rc = 0;		   /* Return code from primary routines	*/

	drive = getdisk();	       /* Get the default current drive	*/

	getcwd(curdir,MAXPATH);	       /* Capture the current directory	*/

	if ( (argc > 1) )		     /* Do we have an argument?	*/
		{
		if ( !(isalpha(argv[1][0])) )
			{
			Instructions();		  /* Tell it like it is	*/
			exit (0);
			}
		else
			{
			drive = toupper(argv[1][0]) - 65;
			}
		}

	if ( (rc = Initialize()) == SUCCESS )
		rc = MainLine();

	setdisk(curdir[0] - 65);	     /* Reset the default drive	*/

	chdir(curdir);	       /* Put the user back to where he/she was	*/

	printf("Directories=%d  Files=%d  Duplicate File Sets=%d\n",dircnt,filcnt,dupcnt);

	if ( (rc == FAIL) )
		abort();

	exit(0);
}

/************************************************************************/
/* User Instructions							*/
/************************************************************************/

int Instructions()
{
	printf("\nEnter FDUPS all by itself at the DOS prompt and it will scan the currently");
	printf("\nactive disk for duplicate file names.\n");

	printf("\nEnter FDUPS followed by at least one space and the letter of the disk you");
	printf("\nwish to scan and FDUPS will scan the designated disk for duplicate file");
	printf("\nnames.\n");

	printf("\nFDUPS will create a file, FDUPS.txt, in the root directory of the disk being");
	printf("\nscanned.  This will overwrite any file already there with the identical");
	printf("\nname.\n");

	printf("\nFDUPS.txt contains the File Name, Last Modified Date and Time, File Size and");
	printf("\nthe fully qualified pathname of each duplicate file.  Each set of duplicate");
	printf("\nfiles is grouped together with a Carriage Return between each group.  FDUPS.txt");
	printf("\nis in DOS text format ready to be viewed with your favorite text editor or");
	printf("\ndisplayed with the DOS TYPE command.\n");

	printf("\nFDUPS will display counts for Directories, Files and Duplicate Sets when");
	printf("\nfinished.\n");

	return (SUCCESS);
}

/************************************************************************/
/* Program Initialization routine					*/
/************************************************************************/

int Initialize()
{
	printf("\n%s\n\n",ProgTitle);

	setdisk(drive);			   /* Point to designated drive	*/

	sprintf(cdrive,"%c:\\",drive + 65);	 /* Make Root Directory	*/

	if ( (chdir(cdrive)) != 0 )		      /* Change to Root	*/
		{
		printf("\n%s ==> %s\n",cdrive,sys_errlist[errno]);
		return (FAIL);
		}
		
	return (SUCCESS);
}

int MainLine()
{
	int ScanDisk(char *dirname);
	int CheckDups();

	row = Row();	   /* Save the current row to display pathnames	*/

	if ( (ScanDisk(cdrive)) == FAIL )    /* Scan the root directory	*/
		return (FAIL);				    /* Bad Luck	*/

	ClearRow(row);		    /* Get rid of last pathname display	*/

	if ( (froot != NULL) )		     /* We must have some files	*/
		if ( (CheckDups()) == FAIL )	/* Check for Duplicates	*/
			return (FAIL);		       /* More bad luck	*/

	return (SUCCESS);
}

int ScanDisk(char *dirname)
{

/* This is a recursive routine performed from within for each directory	*/

	int BuildFileEntry(char *dp,struct ffblk *ffblk);

	int i,a,r,s;
	char temp[MAXPATH],savedir[MAXDIR];
	struct ffblk ffblk;
	char *dp;

	getcwd(savedir,MAXDIR);		  /* Save the current directory	*/

	chdir(dirname);				/* Point to the new one	*/

	s = strlen(dirname);		       /* Get the string length	*/

	if ( (*(dirname + s - 1) != BKSLASH)  )	 /* Check last position	*/
		strcat(dirname,"\\");	      /* Make sure of backslash	*/

	dircnt++;	/* Count the directory	*/

	ClearRow(row);		      /* Jettison the residue on screen	*/
	printf("%s",dirname);		 /* Show the user the path name	*/
	col = Col();	 /* Save the column position for the file names	*/

	if ( (dp = (char *) calloc(1,s)) == NULL )
		{
		printf("\nOut of memory saving path string\n");
		return (FAIL);
		}

	strcpy(dp,dirname);		    /* Store the full path name	*/

	setdta(MK_FP(FP_SEG(&ffblk),FP_OFF(&ffblk)));  /* For Recursion	*/

	a = FA_DIREC | FA_SYSTEM | FA_HIDDEN | FA_RDONLY; /* Attributes	*/

/* Uses the DOS FindFirst/FindNext to scan a directory and its files	*/

	for (i = findfirst("*.*",&ffblk,a),r = 0; i == 0; i = findnext(&ffblk))
		{
		if ( (ffblk.ff_name[0] == '.') )  /* Parent designation	*/
			continue;

		if ( (ffblk.ff_attrib & FA_DIREC) == FA_DIREC )
			{
			strcpy(temp,dirname);  /* Path ending w/bkslash	*/
			strcat(temp,ffblk.ff_name);  /* Append dir name	*/

			if ( (r = ScanDisk(temp)) != 0 ) /* Do it again	*/
				break;
			}
		else
			{
			if ( (r = BuildFileEntry(dp,&ffblk)) == FAIL )
				break;
			}
		}

	if ( (r == 0) )
		chdir(savedir);	/* If all went ok, restore old directory */

	return (r);
}

int BuildFileEntry(char *dp,struct ffblk *ffblk)
{
	char fname[13],tfname[13];
	DFILE far *df,far *tmp,far *prev;

	SetCursor(row,col);		 /* Point to file name position	*/

	printf("%-12s",ffblk->ff_name);	    /* Show it now just in case	*/

/* Need the far heap to store the file structures in the small C model	*/

	if ( (df = (DFILE far *) farcalloc(1,sizeof(DFILE))) == NULL )
		{
		printf("\nOut of memory for filename structure\n");
		return (FAIL);
		}

	filcnt++;	/* Count the file	*/

	movedata(_DS,FP_OFF(ffblk->ff_name),FP_SEG(df->filename),FP_OFF(df->filename),13);

	df->fsize = ffblk->ff_fsize;	 /* Populate the file structure	*/
	df->ftime = ffblk->ff_ftime;
	df->fdate = ffblk->ff_fdate;
	df->parent = dp;

	prev = NULL;

	strcpy(fname,ffblk->ff_name);	   /* Make it easier to compare	*/

/* Run down through the linked list of file names until we find one	*/
/* that's greater. Insert (link) the incoming file name in front of it.	*/

	for (tmp = froot; tmp != NULL; tmp = tmp->next)
		{

	/* Put the file name from the list into a local variable	*/

		movedata(FP_SEG(tmp->filename),FP_OFF(tmp->filename),_DS,FP_OFF(tfname),13);

		if ( (strcmp(tfname,fname)) > 0 )
			{
			if ( (prev == NULL) )
				froot = df;   /* Must be lower than any	*/
			else
				prev->next = df;    /* Successive times	*/

			df->next = tmp;	      /* Add to the linked list	*/
			break;
			}

		if ( (tmp->next == NULL) )
			{
			tmp->next = df;		     /* End of the list	*/
			break;
			}

		prev = tmp;		/* Setup for the next iteration	*/
		}

	if ( (tmp == NULL) )
		froot = df;	/* First time through	*/

	return (SUCCESS);
}

int CheckDups()
{
	BOOL FirstPrinted;
	unsigned pfdate,pftime,cfdate,cftime;
	long pfsize,cfsize;
	char path[MAXPATH],pfname[13],cfname[13];
	FILE *dups;
	DFILE far *tmp;
	char *pdp,*cdp;

	strcpy(path,cdrive);
	strcat(path,"FDUPS.txt");  /* Build the reporting file pathname	*/

	if ( (dups = fopen(path,"wt")) == NULL )       /* Open the file	*/
		{
		printf("\n%s ==> %s\n",path,sys_errlist[errno]);
		return (FAIL);
		}

/* Setup a "prior" set of variables with the first structure's data	*/

	movedata(FP_SEG(froot->filename),FP_OFF(froot->filename),_DS,FP_OFF(pfname),13);

	pfdate = froot->fdate;
	pftime = froot->ftime;
	pfsize = froot->fsize;

	FirstPrinted = FALSE;

/* Scan the file name list starting with the second one	*/

	for (tmp = froot->next, pdp = froot->parent; tmp != NULL; tmp = tmp->next)
		{

       /* Setup a "current" set of variables with this structure's data	*/

		movedata(FP_SEG(tmp->filename),FP_OFF(tmp->filename),_DS,FP_OFF(cfname),13);

		cdp = tmp->parent;
		cfdate = tmp->fdate;
		cftime = tmp->ftime;
		cfsize = tmp->fsize;

	/* Compare the previous and current file names	*/

		if ( (strcmp(pfname,cfname)) == 0 )	   /* Duplicate	*/
			{
			if ( !(FirstPrinted) )	    /* First of the set	*/
				{
				fprintf(dups,"\n%-12s %s %s % 8li %s\n",pfname,EditDate(pfdate),EditTime(pftime),pfsize,pdp);
				FirstPrinted = TRUE;
				dupcnt++;	     /* Count only sets	*/
				}

			fprintf(dups,"%-12s %s %s % 8li %s\n",cfname,EditDate(cfdate),EditTime(cftime),cfsize,cdp);
			}
		else
			FirstPrinted = FALSE;

		strcpy(pfname,cfname);  /* Setup next "prior" variables	*/
		pdp = cdp;
		pfdate = cfdate;
		pftime = cftime;
		pfsize = cfsize;
		}

	fclose(dups);			    /* Close the reporting file	*/

	return (SUCCESS);
}

/************************************************************************/
/* Global Utility Functions 						*/
/************************************************************************/

char *EditDate(unsigned date)
 {
 	static char xdate[9];

	sprintf(xdate,"%02d-%02d-%02d",(date % 512) / 32,(date % 512) % 32,(date / 512) + 80);

 	return (xdate);
 }

char *EditTime(unsigned time)
{
	int hr,min;
	unsigned ampm;
	static char xtime[7];

	if ( (hr = time / 2048) > 11 )	      /* Separate out the hours	*/
		{
		ampm = 'p';			  /* Determine am or pm	*/

		if ( (hr > 12) )
			hr -= 12;
		}
	else
		ampm = 'a';

	min = (time % 2048) / 32;	    /* Separate out the minutes	*/

	sprintf(xtime,"%2d:%02d%c",hr,min,ampm);

	return (xtime);
}

void ClearRow(int row)	/* Blanks out the specified row by scrolling	*/
{
	_BH = 7;		/* Normal Attribute		*/
	_CH = row;		/* Top Row of window		*/
	_CL = 0;		/* Left Col of window		*/
	_DH = row;		/* Bottom Row of window		*/
	_DL = 79;		/* Right Col of window		*/
	_AL = 1;		/* Lines to scroll		*/
	_AH = 6;		/* Scroll up Function Code	*/

	geninterrupt(0x10);

	SetCursor(row,0);		    /* Position at start of row	*/
}

void SetCursor(int row,int col)
{
	_DL = col;
	_DH = row;
	_AH = 2;			 /* Function code to set cursor	*/
	_BH = 0;				/* Current display page	*/

	geninterrupt(0x10);
}

unsigned Row()				/* Gets the current display row	*/
{
	_AH = 3;			      /* Function to get cursor	*/
	_BH = 0;			        /* Current display page	*/

	geninterrupt(0x10);

	return (_DH);		       /* The row is in the dh register	*/
}

unsigned Col()			     /* Gets the current display column	*/
{
	_AH = 3;			      /* Function to get cursor	*/
	_BH = 0;			        /* Current display page	*/

	geninterrupt(0x10);

	return (_DL);		    /* The column is in the dl register	*/
}
