/****************************************************************************/
/*                                                                          */
/* adbload.c                                                                */
/*                                                                          */
/* Read ASCII data and import it into HP100LX .ADB file                     */
/*                                                                          */
/* A. Garzotto, April '94                                                   */
/*                                                                          */
/* Updated for Win32 by James Edward Lewis II, 20 September 2009            */
/*                                                                          */
/****************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <cstring>
#include <ctype.h>
#include <fcntl.h>

#define YES 1
#define NO  0
#define NIL 0

#ifndef O_BINARY
#define O_BINARY 0
#endif

#define S_ALARM     1
#define S_CHECKOFF  2
#define S_MONTHLY   2
#define S_CARRY     4
#define S_WEEKLY    4
#define S_TODO     16
#define S_EVENT    32
#define S_STUB     64
#define S_APPT    128

/****************************************************************************/

struct rep_desc
{
   char freq;
   int days;
   int month;
   char y1;
   char m1;
   char d1;
   char y2;
   char m2;
   char d2;
   char ndeleted;
   char *deleted;
};

typedef struct rep_desc *REPEAT;

struct rec_desc
{
   int num;                 /* number of this record */
   char reptype;            /* repeat type */
   char state;              /* record state */
   char *desc;              /* description */
   char *location;          /* location */
   char year;               /* start date; */
   char month;
   char day;
   int stime;               /* start time */
   int etime;               /* end time */
   int duration;            /* durations in days */
   char prio[2];            /* todo priority */
   int lead;                /* alarm lead time */
   char *note;              /* note */
   REPEAT repeat;           /* pointer to repeat description */
   struct rec_desc *next;   /* pointer to next element in list */
};

typedef struct rec_desc *RECORD;

/****************************************************************************/

FILE *fin;               /* CDF file */
int handle;             /* resulting .ADB file */

RECORD records = NIL;    /* list of data records */
int numrecords = 0;      /* number of records in database */
int notenum = 0;         /* number of current note */

int debug = NO;          /* debug mode */
char dateformat[80] = "d.m.yyyy"; /* date input format */
int retain_crlf = NO;    /* expect CR/LF in text fields */
int tquotes = YES;       /* expect quotes around text fields */
int nquotes = NO;        /* expect quotes around number fields */

char *monthnames[] =
{
   "JAN",
   "FEB",
   "MAR",
   "APR",
   "MAY",
   "JUN",
   "JUL",
   "AUG",
   "SEP",
   "OCT",
   "NOV",
   "DEC"
};

/****************************************************************************/
/* allocate memory */

char *my_malloc(int size)
{
   char *p;

   p = (char *)malloc(size);
   if (!p)
   {
      fprintf(stderr, "Memory allocation problems.\nAborted!\n");
      exit(1);
   }
   return p;
}

/****************************************************************************/
/* create a new record struct */

RECORD new_rec()
{
   RECORD r;

   r = (RECORD)my_malloc(sizeof(struct rec_desc));
   r->reptype = 0;
   r->year = r->month = r->day = r->state = 0;
   r->stime = r->etime = r->duration = r->lead = 0;
   r->desc = r->location = r->note = NIL;
   r->next = records;
   records = r;
   r->num = numrecords++;
   r->repeat = NIL;
   return r;
}

/****************************************************************************/
/* create a new repeat struct */

REPEAT new_rep()
{
   REPEAT r;

   r = (REPEAT)my_malloc(sizeof(struct rep_desc));
   r->freq = r->y1 = r->m1 = r->d1 = r->y2 = r->m2 = r->d2 = '\0';
   r->ndeleted = '\0';
   r->deleted = NIL;
   r->days = r->month = 0;
   return r;
}

/****************************************************************************/
/* create a new .ADB file header */

void put_header()
{
   static char infostruct[]=
   {
   0x0e, 0x02, 0x1d, 0x00, 0x00, 0x00,
   0x5e, 0x09, 0x17, 0xe4, 0x0f, 0x14, 0x00, 0x06, 0x05, 0xe0, 0x01, 0x3c,
   0x00, 0x05, 0x00, 0x31, 0x00, 0xff, 0xff, 0xFF, 0xFF, 0x03, 0x00
   };

   char buf[29];

   buf[0] = 'h';
   buf[1] = 'c';
   buf[2] = 'D';
   buf[3] = '\0';

   buf[4] = '\0'; /* type header block */
   buf[5] = '\0'; /* status: OK */
   buf[6] = (char)25; /* length */
   buf[7] = '\0';
   buf[8] = buf[9] = '\0'; /* index */

   buf[10] = (int)2; /* Release */
   buf[11] = (int)1;
   buf[12] = '2'; /* type apptbook */
   buf[13] = '\0'; /* file status */
   buf[14] = buf[15] = (char)0x00; /* current viewpoint */
   buf[16] = (char)(numrecords % 256); /* number of records */
   buf[17] = (char)(numrecords / 256);
   buf[18] = buf[19] = buf[20] = buf[21] = '\0'; /* no lookup table */
   buf[22] = buf[23] = buf[24] = buf[25] = buf[26] = '\0'; /* time stamp */
   buf[27] = 0x37;
   buf[28] = 0x84; /* magic hash code */
   write(handle, buf, 29);

   buf[0] = (char)0x0A; /* type viewpoint table */
   buf[1] = '\0'; /* status: ok */
   buf[2] = (char)8; /* length */
   buf[3] = '\0';
   buf[4] = buf[5] = '\0'; /* index */

   buf[6] = buf[7] = (char)0xFF; /* invalid entry */

   write(handle, buf, 8);
   write(handle, infostruct, 29);
}

/****************************************************************************/
/* check if file format is ok */

void check_syntax(char *buf, char *p, char ch, char *msg)
{
   int i;

   if (*p == ch) return;
   if (!*p || ((p > buf) && !p[-1]))
      fprintf(stderr, "Syntax error %s: unexpected end of line!\n", msg);
   else if (ch != '?')
      fprintf(stderr, "Syntax error %s: `%c' expected!\n", msg, ch);
   else
      fprintf(stderr, "Syntax error %s: unexpected character!\n", msg);
   ch = *p;
   *p = '\0';
   fprintf(stderr, "%s<***HERE***>%c%s", buf, ch, &p[1]);
   exit(1);
}

/****************************************************************************/
/* read number from ascii string */

int get_number(char **p)
{
   int n;

   if (nquotes) check_syntax(*p, (*p)++, '\"', "before number");
   sscanf(*p, "%d", &n);
   while (**p && (**p >= '0') && (**p <= '9')) (*p)++;
   if (nquotes) check_syntax(*p, (*p)++, '\"', "after number");
   return n;
}

/****************************************************************************/
/* compare two strings ignoring case */

int my_strncasecmp(char *s1, char *s2, int n)
{
   while (*s1 && *s2 && n && (toupper(*s1) == toupper(*s2)))
   {
      s1++; s2++; n--;
   }
   if (n) return YES;
   return NO;
}

/****************************************************************************/
/* read a date */

void get_date(char **p, char *y, char *m, char *d)
{
   char *f = dateformat;
   char *s = *p;
   int i, nq = nquotes;

   nquotes = NO;
   while (*f)
   {
      switch (*f)
      {
      case 'd': while (*f == 'd') f++;
                *d = (char)(get_number(p) - 1);
                break;
      case 'm': i = 0;
                while (*f == 'm') { f++; i++; }
                if (i > 2)
                {
                   i = 0;
                   while ((i < 12) && my_strncasecmp(*p, monthnames[i], 3))
                      i++;
                   if (i >= 12)
                      check_syntax(s, *p, '?', "in month name");
                   else
                      *m = (char)i;
                   (*p) += 3;
                }
                else
                   *m = (char)(get_number(p) - 1);
                break;
      case 'y': while (*f == 'y') f++;
                i = get_number(p);
                if (i >= 100)
                   *y = (char)(i - 1900);
                else
                   *y = (char)i;
                break;
      default: check_syntax(s, *p, *f, "in 'date'");
               (*p)++; f++;
               break;
      }
   }
   nquotes = nq;
}

/****************************************************************************/
/* read a time field */

int get_time(char **p)
{
   int h, m;
   char *buf = *p;

   if (tquotes) check_syntax(buf, (*p)++, '\"', "before 'time'");
   sscanf(*p, "%d:%d", &h, &m);
   while (**p && (**p != '\"') && (tquotes || (**p != ','))) (*p)++;
   if (tquotes) check_syntax(buf, (*p)++, '\"', "after 'time'");
   return (60 * h + m);
}

/****************************************************************************/
/* read a text or note field */

void get_text(char **p, char **str)
{
   char *q, *qq, *qend;
   char *buf = *p;

   if (tquotes) check_syntax(buf, (*p)++, '\"', "before text field");

   q = *p;
   while (*q)
   {
      if (retain_crlf && (*q == '\n') && (q != *p))
         fgets(&q[1], 1024, fin);
      if (tquotes && (*q == '\"') && (q[-1] != '\\')) break;
      if (!tquotes && (*q == ',') && (q[-1] != '\\')) break;
      if (!tquotes && ((*q == '\n') || (*q == '\r'))) break;
      q++;
   }
   qend = q;

   *str = (char *)my_malloc(qend - *p + 2);
   q = *p;
   qq = *str;
   while (q < qend)
   {
      if (*q == '\\')
      {
         q++;
         switch (*q)
         {
         case 'r': *(qq++) = '\r'; break;
         case 'n': *(qq++) = '\n'; break;
         default: *(qq++) = *q;
         }
         q++;
      }
      else
         *(qq++) = *(q++);
   }
   *qq = '\0';
   *p = qend;
   if (tquotes) check_syntax(buf, (*p)++, '\"', "after text field");
}


/****************************************************************************/
/* add records from CDF file */

void load_new_records()
{
   char *buf;
   char *p;
   RECORD r;
   int num, i;

   buf = (char *)my_malloc(8194);
   while (fgets(buf, 8192, fin))
   {
      if (*buf > ' ')
      {
         r = new_rec();
         p = buf;
         if (tquotes) check_syntax(buf, p++, '\"', "(is this a CDF file?)");
         get_date(&p, &(r->year), &(r->month), &(r->day));
         if (tquotes) check_syntax(buf, p++, '\"', "after 'date'");
         check_syntax(buf, p++, ',', "after 'date'");
         get_text(&p, &(r->desc));
         check_syntax(buf, p++, ',', "after 'description'");
         get_text(&p, &(r->location));
         check_syntax(buf, p++, ',', "after 'location'");
         r->state = (char)get_number(&p);
         check_syntax(buf, p++, ',', "after 'state'");

         if (r->state & S_APPT)
         {
            r->stime = get_time(&p);
            check_syntax(buf, p++, ',', "after 'start time'");
            r->etime = get_time(&p);
            check_syntax(buf, p++, ',', "after 'end time'");
            r->duration = get_number(&p);
            check_syntax(buf, p++, ',', "after 'duration'");
            r->lead = get_number(&p);
            check_syntax(buf, p++, ',', "after 'lead time'");
         }
         else if (r->state & S_EVENT)
         {
            r->duration = get_number(&p);
            check_syntax(buf, p++, ',', "after 'duration'");
            r->stime = r->etime = -1;
            r->lead = 0;
         }
         else /* TODO */
         {
            if (tquotes) check_syntax(buf, p++, '\"', "before 'priority'");
            r->prio[0] = *(p++);
            r->prio[1] = *(p++);
            if (tquotes) check_syntax(buf, p++, '\"', "after 'priority'");
            check_syntax(buf, p++, ',', "after 'priority'");
            r->duration = get_number(&p);
            check_syntax(buf, p++, ',', "after 'duration'");
         }
         get_text(&p, &(r->note));
         if (*p == ',')
         {
            p++;
            r->reptype = (char)get_number(&p);
            check_syntax(buf, p++, ',', "after 'repeat type'");
            r->repeat = new_rep();
            r->repeat->freq = (char)get_number(&p);
            check_syntax(buf, p++, ',', "");
            r->repeat->days = get_number(&p);
            check_syntax(buf, p++, ',', "");
            r->repeat->month = get_number(&p);
            check_syntax(buf, p++, ',', "");
            r->repeat->y1 = (char)get_number(&p);
            check_syntax(buf, p++, ',', "");
            r->repeat->m1 = (char)get_number(&p);
            check_syntax(buf, p++, ',', "");
            r->repeat->d1 = (char)get_number(&p);
            check_syntax(buf, p++, ',', "");
            r->repeat->y2 = (char)get_number(&p);
            check_syntax(buf, p++, ',', "");
            r->repeat->m2 = (char)get_number(&p);
            check_syntax(buf, p++, ',', "");
            r->repeat->d2 = (char)get_number(&p);
            check_syntax(buf, p++, ',', "");
            r->repeat->ndeleted = (char)get_number(&p);
            num = (int)r->repeat->ndeleted & 255;
            if (num)
            {
               check_syntax(buf, p++, ',', "");
               r->repeat->deleted = (char *)my_malloc(4 * num);
               for (i = 0; i < 4 * num; i++)
               {
                  check_syntax(buf, p - 1, ',', "in 'deleted record list'");
                  r->repeat->deleted[i] = (char)get_number(&p);
                  p++;
               }
            }
         }
         else
            r->reptype = (char)1;
      }
   }
   fclose(fin);
   free(buf);
}

/****************************************************************************/
/* write a data record to the .ADB file */

void write_record(RECORD r)
{
   char header[6];
   char data[1024];
   int len, tlen, x, i;

   len = 30 + strlen(r->desc) + strlen(r->location);
   tlen = len;
   if (r->repeat)
      tlen += 12 + 4 * ((int)r->repeat->ndeleted & 255);

   data[0] = (char)(tlen % 256);
   data[1] = (char)(tlen / 256);

   x = 27 + strlen(r->desc) + 1;
   data[2] = (char)(x % 256);
   data[3] = (char)(x / 256);

   data[4] = data[2] + 1;
   data[5] = data[3];

   x = len;
   data[6] = (char)(x % 256);
   data[7] = (char)(x / 256);

   if (*(r->note))
   {
      data[8] = (char)(notenum % 256);
      data[9] = (char)(notenum / 256);
   }
   else
      data[8] = data[9] = (char)255;

   data[10] = (char)255; /* previous rec on same day */
   data[11] = (char)255;
   data[12] = (char)255; /* next rec on same day */
   data[13] = (char)255;

   data[14] = r->state;

   data[15] = r->year;
   data[16] = r->month;
   data[17] = r->day;

   x = r->stime;
   if (x == -1)
   {
      data[18] = (char)255;
      data[19] = (char)255;
   }
   else
   {
      data[18] = (char)(x % 256);
      data[19] = (char)(x / 256);
   }
   x = r->duration;
   data[20] = (char)(x % 256);
   data[21] = (char)(x / 256);
   x = r->etime;
   if (x == -1)
   {
      data[22] = (char)255;
      data[23] = (char)255;
   }
   else
   {
      data[22] = (char)(x % 256);
      data[23] = (char)(x / 256);
   }

   x = r->lead;
   data[24] = (char)(x % 256);
   data[25] = (char)(x / 256);
   if (r->state & S_TODO)
   {
      data[18] = r->prio[0];
      data[19] = r->prio[1];
      data[22] = data[23] = data[24] = data[25] = (char)255;
   }

   data[26] = r->reptype;
   strcpy(&data[27], r->desc);
   x = 28 + strlen(r->desc);
   data[x] = '\0';
   x++;
   strcpy(&data[x], r->location);
   if (r->repeat)
   {
      x = len;
      data[x++] = r->repeat->freq;
      data[x++] = (char)(r->repeat->days % 256);
      data[x++] = (char)(r->repeat->days / 256);
      data[x++] = (char)(r->repeat->month % 256);
      data[x++] = (char)(r->repeat->month / 256);
      data[x++] = r->repeat->y1;
      data[x++] = r->repeat->m1;
      data[x++] = r->repeat->d1;
      data[x++] = r->repeat->y2;
      data[x++] = r->repeat->m2;
      data[x++] = r->repeat->d2;
      data[x++] = r->repeat->ndeleted;
      for (i = 0; i < 4 * ((int)r->repeat->ndeleted & 255); i++)
         data[x++] = r->repeat->deleted[i];
   }

   header[0] = (char)11; /* type data record */
   header[1] = (char)0; /* status OK */
   header[2] = (char)((tlen + 6) % 256); /* length of record */
   header[3] = (char)((tlen + 6) / 256);
   header[4] = (char)(r->num % 256);     /* record number */
   header[5] = (char)(r->num / 256);
   write(handle, header, 6);
   write(handle, data, tlen);
}

/****************************************************************************/
/* write a note record to the .ADB file */

void write_note(char *n)
{
   char header[6];
   int len;

   len = strlen(n);

   header[0] = (char)9;
   header[1] = (char)0;
   header[2] = (char)((len + 6) % 256);
   header[3] = (char)((len + 6) / 256);
   header[4] = (char)(notenum % 256);
   header[5] = (char)(notenum / 256);

   write(handle, header, 6);
   write(handle, n, len);
   notenum++;
}

/****************************************************************************/
/* add new records to .ADB file */

void write_new_records()
{
   RECORD r = records;

   while (r)
   {
      write_record(r);
      if (*(r->note))
         write_note(r->note);
      r = r->next;
   }
   close(handle);
}

/****************************************************************************/
/* display usage information */

void help(char *prog)
{
   fprintf(stderr, "ADBLOAD version 1.3 by Andreas Garzotto\n\n");
   fprintf(stderr, "Update for Win32 by James Edward Lewis II\n\n");
   fprintf(stderr,
      "USAGE: %s [options] <CDF file> <.ADB file> \n", prog);
   fprintf(stderr, " Options:\n");
   fprintf(stderr,
      "  -d fmt expect date as specified in fmt (default: 'd.m.yyyy')\n");
   fprintf(stderr, "  -q0    do not expect quotes around any fields\n");
   fprintf(stderr,
      "  -q1    expect quotes around text fields only (default)\n");
   fprintf(stderr,
      "  -q2    expect quotes around text fields and numbers\n");
   fprintf(stderr,
      "  -r     expect CR/LF in note fields (not '\\r\\n')\n");

   exit(1);
}

/****************************************************************************/
/* decode options */

void get_options(int argc, char **argv)
{
   int state = 0, i = 1;

   while (i < argc)
   {
      if (argv[i][0] == '-')
      {
         switch (argv[i][1])
         {
         case 'x': debug = YES; break;
         case 'd': if (i >= argc - 1) break;
                   strcpy(dateformat, argv[++i]);
                   break;
         case 'q': if (argv[i][2] == '0') tquotes = NO;
                   if (argv[i][2] == '2') nquotes = YES;
                   break;
         case 'r': retain_crlf = YES; break;
         default: help(argv[0]);
         }
      }
      else
      {
         if (state == 0)
         {
            fin = fopen(argv[i], "r");
            if (fin == NULL)
            {
               fprintf(stderr, "Cannot open input file '%s'\n", argv[i]);
               exit(1);
            }
            state = 1;
         }
         else if (state == 1)
         {
            handle = open(argv[i], O_TRUNC|O_CREAT|O_RDWR|O_BINARY, 420);
            if (handle == -1)
            {
               fprintf(stderr, "Cannot open output file '%s'\n", argv[i]);
               exit(1);
            }
            state = 2;
         }
         else
            help(argv[0]);
      }
      i++;
   }
}

/****************************************************************************/

int main(int argc, char **argv)
{
   if (argc < 3) help(argv[0]);

   get_options(argc, argv);
   load_new_records();
   put_header();
   write_new_records();
   return 0;
}

/****************************************************************************/
