/*****************************************************************************/
/** PROJECT: Buzzword Bingo                                                 **/
/**                                                                         **/
/**    FILE: BUZZBING.C                                                     **/
/**                                                                         **/
/** PURPOSE: Generator for "Buzzword Bingo" cards                           **/
/**                                                                         **/
/** CREATED: 10/15/1998                                                     **/
/**                                                                         **/
/** HISTORY:                                                                **/
/**                                                                         **/
/**   NOTES:                                                                **/
/**                                                                         **/
/*****************************************************************************/
#define VERSION "1.00  "
#define LASTMOD "10/15/1998"
#define COPYRIGHT "Copyright 1998, Steven L. Van Dyke. All Rights Reserved"

#include <stdio.h>
#include <stdlib.h>
#include <dir.h>


/** Defines **/

#define TRUE   1
#define FALSE  0

#define DEFAULT_WORDS "buzzbing.txt"
#define DEFAULT_CARDS 1
#define WHEREGO       "buzzcard.txt"

   /* Drawing characters */
#define ULC        201
#define TOP_BAR    205
#define TOP_INT    203
#define URC        187
#define LEFT_SIDE  186
#define LS_INT     204
#define RIGHT_SIDE 186
#define RS_INT     185
#define MID_BAR    205
#define MID_VERT   186
#define MID_INT    206
#define LLC        200
#define BOT_BAR    205
#define BOT_INT    202
#define LRC        188

/** Enums **/

enum { TOP_ROW, MID_ROW, BOT_ROW };  /* Row Types */

/** Typedefs **/

/** Prototypes **/

int  main( int argc, char **argv);
int  get_words( char *szWordFile );
void pick_words( void );
void print_card( void );
void print_bar( int sBarType );
void print_row( int sRow );
void fill_cell( int sCell, char *szWord );

/** Global Data **/

char CopyRightMsg[] = COPYRIGHT;

char **aszWordList;
char *szWords4Card[25];
char szFree[] = "FREE";

char szCells[5][ 8 * 14 + 1 ]; /* Array of cells for a row */

int sNumWords;
int  fFree;     /* We have a 'FREE' space flag */

FILE *fpWords;
FILE *fpOut;

/*****************************************************************************/
/** FUNCTION: main()                                                        **/
/**                                                                         **/
/**  PURPOSE: Main module of program                                        **/
/**                                                                         **/
/**    INPUT:                                                               **/
/**           int argc - number of arguments                                **/
/**           char **argv - array of argument strings                       **/
/**                                                                         **/
/**   OUTPUT: Error code / 0 if no error                                    **/
/**                                                                         **/
/**  HISTORY:                                                               **/
/**                                                                         **/
/**    NOTES:                                                               **/
/**                                                                         **/
/*****************************************************************************/
int main( int argc, char **argv)
{
   char szDrive[MAXDRIVE],
        szDir[MAXDIR],
        szName[MAXFILE],
        szExt[MAXEXT],
        szWordFile[MAXDRIVE+MAXDIR+MAXFILE+MAXEXT],
        szOutFile[MAXDRIVE+MAXDIR+MAXFILE+MAXEXT],
        szTmpFile[MAXDRIVE+MAXDIR+MAXFILE+MAXEXT],
        szTmp[20];
   int nextarg, fHelp, sCards, i, j;

   randomize();
   strcpy( szWordFile, DEFAULT_WORDS );    /* Set up default word file name */
   sCards = DEFAULT_CARDS;                 /* Default # of cards */
   strcpy( szOutFile, WHEREGO );

   printf("Buzzword Bingo card Generator (BUZZBING), %s %s\n", VERSION, LASTMOD);
   nextarg = 1;
   fHelp  = FALSE;                /* Assume don't want help */
   fFree = TRUE;                  /* Assume there *is* a free space */
   while ((argv[nextarg][0] == '-' || argv[nextarg][0] == '/')
            && argv[nextarg] != NULL)
   {
      if (argv[nextarg][1] == '?')
         fHelp = TRUE;
      else if ( ( argv[nextarg][1] & 0xDF ) == 'F' )
         fFree = FALSE;
      ++nextarg;
   }
   if ( argc < 1 || fHelp )
   {
      printf("Syntax is BUZZBING [-opt] <filespec> <cards> <output to>\n"
           "[opt] is one of the following options:\n"
           "     -? = This message\n"
           "     -f = No free space\n"
           "<filespec> - Name of buzzword file (%s is default)\n"
           "<cards>    - Number of cards to generate (%d is default)\n"
           "<output to>- Where to 'print' the cards (%s is default)\n"
           "   <Press Enter to continue>\n",
           szWordFile, sCards, szOutFile );
      getchar();
      printf( "This program generates game cards for Buzzword Bingo.\n"
                "   Buzzword Bingo is the game that makes otherwise boring "
                "meetings worth\n   your time.  Distribute the cards to the "
                "players before the meeting,\n   decide on the rules (regular,"
                " 4 corners, are buzzwords in handouts allowed,\n   etc.).  "
                "Now you'll have an incentive to stay awake and attentive.\n"
                "To use this program, you need to provide a file of "
                "buzzwords for your\n   environment, and the number of cards "
                "to generate.  The list of words must\n   be in a file with "
                "the number of words on the first line, and each of the\n"
                "   rest of the lines containing one 'buzzword'.  Note that "
                "each buzzword can\n   actually be up to 112 characters long."
                "  The program won't try to break\n   phrases at spaces.  "
                "You can force a line break with a '|' (vertical bar).\n"
                "   You *must* have at least 25 words, and for best results "
                "you want 75 or so.\n"
           "BUZZBING is copyrighted Postcard ware.  If you like it, "
             "send a *picture*\n"
           "  postcard from where you are (neat stamps appreciated) to:\n"
           "                 Steven Van Dyke\n"
           "                 7918 N. Central\n"
           "                 Kansas City, MO USA 64118\n"
         );
      return(0);         /* exit with no error */
   }
   
   /* Get the file spec (if there is one) */
   
   if ( argc > nextarg )
   {
      strcpy( szWordFile, argv[ nextarg ] );
      ++nextarg;
   }
   
   if ( argc > nextarg )
   {
      sCards = atoi( argv[ nextarg ] );
      ++nextarg;
   }
   
   if ( argc > nextarg )
   {
      strcpy( szOutFile, argv[ nextarg ] );
      ++nextarg;
   }
   
   if ( access( szWordFile, 4 ) != 0 )
   {
      printf( "Unable to access buzzword file '%s' for read!\n"
              "Aborting...\n", szWordFile );
      exit( -1 );
   }
   
   if ( sCards < 1 )
   {
      printf( "Invalid number of cards (%d).  Resetting to %d\n", 
               sCards, DEFAULT_CARDS );
      sCards = DEFAULT_CARDS;
   }

   /****************************/
   /**  Open the Output file  **/
   /****************************/
   
   if ( ( fpOut = fopen( szOutFile, "wt" ) ) == NULL )
   {
      printf( "Unable to open '%s' for output!\n", szOutFile );
      fclose( fpWords );
      exit( -1 );
   }
   
   /*************************/
   /**  Get the word list  **/
   /*************************/
   
   if ( (i = get_words( szWordFile ) ) != 0 )
   {
      fclose( fpOut );
      exit( i );
   }
   
   /***********************/
   /**  Print the cards  **/
   /***********************/

   for ( i = 0; i < sCards; ++i )
   {
      print_card();      
   } /* end for all cards */
   return( 0 );
}

/*****************************************************************************/
/** FUNCTION: GET_WORDS                                                     **/
/**                                                                         **/
/**  PURPOSE: Read in buzzword list file                                    **/
/**                                                                         **/
/**    INPUT: char *szWordFile - Name of word list file                     **/
/**                                                                         **/
/**   OUTPUT: int serr - Error code (0 if no error)                         **/
/**                                                                         **/
/**  HISTORY:                                                               **/
/**                                                                         **/
/**    NOTES:                                                               **/
/**           This function assumes the word file has one entry / line.     **/
/**           The first line should be the number of words in the file.     **/
/**                                                                         **/
/**           Example:                                                      **/
/**           5                                                             **/
/**           Proactive                                                     **/
/**           Agent                                                         **/
/**           Synergy                                                       **/
/**           Web-Centric                                                   **/
/**           Total Cost of Ownership                                       **/
/**                                                                         **/
/**           Note that entries can be up to 112 characters long.  Lines    **/
/**           will be broken at spaces if possible, or line breaks can      **/
/**           be forced by using |                                          **/
/**                                                                         **/
/*****************************************************************************/
int  get_words( char *szWordFile )
{
   int   i, j;
   char  szLine[120];
      
   if ( ( fpWords = fopen( szWordFile, "rt" ) ) == NULL )
   {
      printf( "Error %d opening word file!\n", errno );
      return( -1 );
   }
   
   if ( fgets( szLine, sizeof( szLine ) - 1, fpWords ) == NULL )
   {
      printf( "Error %d reading number of words!\n", errno );
      fclose( fpWords );
      return( -2 );
   }
   
   sNumWords = atoi( szLine );
   if ( sNumWords < 25 )
   {
      printf( "I need at least 25 words to work with!\n"
              "You said there were only %d words in the file.\n",
              sNumWords );
      fclose( fpWords );
      return( -3 );
   }
   
   /**************************************************/
   /** Allocate array of string pointers for words  **/
   /**************************************************/

   aszWordList = malloc( sNumWords * sizeof( char * ) );
   
   if ( aszWordList == NULL )
   {
      printf( "Unable to allocate memory for word list.\n" );
      fclose( fpWords );
      return( -4 );
   }
   
   for ( i = 0; i < sNumWords; ++i )
   {
      if ( fgets( szLine, sizeof( szLine ) - 1, fpWords ) == NULL )
      {
         printf( "Ran out of words at line %d!\n", i+2 );
         fclose( fpWords );
         
         if ( i < 25 )
         {
            printf( "That's not enough words to continue!\n" );
            return( -5 );
         }
         break; /* exit loop */
      }
      
      szLine[119] = '\0'; /* Make sure we're terminated */
      j = strlen( szLine ) - 1;   /* Trim off trailing \n */
      if ( szLine[j] == '\n' )
         szLine[j] = '\0';
      
      if ( ( aszWordList[i] = malloc( strlen( szLine ) + 1 ) ) == NULL )
      {
         printf( "Unable to allocate memory for word on line %d!\n", i + 2 );
         fclose( fpWords );
         return( -6 );
      }
      
      strcpy( aszWordList[i], szLine );
   
   } /* end for reading all words in file */
   
   fclose( fpWords );   /* Done with file */
   return( 0 );
}

/*****************************************************************************/
/** FUNCTION: PICK_WORDS()                                                  **/
/**                                                                         **/
/**  PURPOSE: Pick up the set of words for this card                        **/
/**                                                                         **/
/**    INPUT: (void)                                                        **/
/**                                                                         **/
/**   OUTPUT: (void)                                                        **/
/**                                                                         **/
/**  HISTORY:                                                               **/
/**                                                                         **/
/**    NOTES:                                                               **/
/**           This function picks a random set of 24 words (plus "FREE"),   **/
/**           making sure there are no repeated words.  The resulting       **/
/**           word list is pointed to by szWords4Card                       **/
/**                                                                         **/
/*****************************************************************************/
void pick_words( void )
{
   int   sRand;       /* Current random number */
   int   sNums[25];   /* List of selected numbers */
   int   fUsed;
   int   i, j;
   
   for ( i = 0; i < 25; ++i )
      sNums[i] = -1;  /* Set to 'unchosen' */
      
   for ( i = 0; i < 25; ++i )
   {
      if ( i == 12 && fFree  )
         continue;      /* Skip 'FREE' space */
      
      do
      {
         sRand = rand() % sNumWords;
         
         for ( j = 0, fUsed = FALSE; j < i; ++j )
         {
            if ( sRand == sNums[j] )
            {
               fUsed = TRUE;
               break;
            }
         }
      } while ( fUsed );   /* Go 'til it's unique */

      sNums[i] = sRand;

   } /* end while getting a set of words */
   
   for ( i = 0; i < 25; ++i )
   {
      if ( i == 12 && fFree )   /* "FREE" is a bit different */
         szWords4Card[i] = szFree;
      else
         szWords4Card[i] = aszWordList[ sNums[i] ];
   } /* end for picking up actual words */
}
         
/*****************************************************************************/
/** FUNCTION: PRINT_CARD()                                                  **/
/**                                                                         **/
/**  PURPOSE: Print out a single buzzword bingo card                        **/
/**                                                                         **/
/**    INPUT: (void)                                                        **/
/**                                                                         **/
/**   OUTPUT: (void)                                                        **/
/**                                                                         **/
/**  HISTORY:                                                               **/
/**                                                                         **/
/**    NOTES:                                                               **/
/**                                                                         **/
/*****************************************************************************/
void print_card( void )
{
   int i;

   /************************************/
   /**  Pick the words for this card  **/
   /************************************/

   pick_words();
      
   /************************/
   /**  Print the Top Bar **/
   /************************/
   
   print_bar( TOP_ROW );

   /*******************/
   /**  Print row 1  **/
   /*******************/
   
   print_row( 0 );
      
   for ( i = 1; i < 5; ++i )
   {
            
      /***********************/
      /**  Print a mid bar  **/
      /***********************/
      
      print_bar( MID_ROW );
      
      /*****************/
      /**  Print row  **/
      /*****************/
    
      print_row( i );
           
   } /* end for rest of rows */

   /****************************/
   /**  Print the bottom bar  **/
   /****************************/
      
   print_bar( BOT_ROW );

   fprintf( fpOut, "%c", 12 );   /* Trailing <FF> */      
}


/*****************************************************************************/
/** FUNCTION: PRINT_BAR()                                                   **/
/**                                                                         **/
/**  PURPOSE: Print out one of the lines (bars) of the grid                 **/
/**                                                                         **/
/**    INPUT: int sBarType - which type of bar (top, mid, bottom)           **/
/**                                                                         **/
/**   OUTPUT: (void)                                                        **/
/**                                                                         **/
/**  HISTORY:                                                               **/
/**                                                                         **/
/**    NOTES:                                                               **/
/**                                                                         **/
/*****************************************************************************/
void print_bar( int sBarType )
{
   char szLine[77];
   char cLeft, cMid, cInt, cRight;
   
   switch ( sBarType )
   {
   case ( TOP_ROW ):
      cLeft  = ULC;
      cMid   = TOP_BAR;
      cInt   = TOP_INT;
      cRight = URC;
      break;
   case ( MID_ROW ):
      cLeft  = LS_INT;
      cMid   = MID_BAR;
      cInt   = MID_INT;
      cRight = RS_INT;
      break;
   case ( BOT_ROW ):
      cLeft  = LLC;
      cMid   = BOT_BAR;
      cInt   = BOT_INT;
      cRight = LRC;
      break;
   } /* end which type of bar */
   
   memset( szLine, cMid, 76 );   /* Fill line with mid-line character */
   szLine[0]  = cLeft;    /* Add the special characters in their spots */
   szLine[15] = cInt;
   szLine[30] = cInt;
   szLine[45] = cInt;
   szLine[60] = cInt;
   szLine[75] = cRight;
   szLine[76] = '\0';

   fprintf( fpOut, "  %s\n", szLine );  /* Send to output */
}

/*****************************************************************************/
/** FUNCTION: PRINT_ROW()                                                   **/
/**                                                                         **/
/**  PURPOSE: Print out a row of the bingo card                             **/
/**                                                                         **/
/**    INPUT: int sRow - which row (0 - 4) to print                         **/
/**                                                                         **/
/**   OUTPUT: (void)                                                        **/
/**                                                                         **/
/**  HISTORY:                                                               **/
/**                                                                         **/
/**    NOTES:                                                               **/
/**            This function loads up the words into the szCells array      **/
/**               and then prints that out.                                 **/
/**                                                                         **/
/*****************************************************************************/
void print_row( int sRow )
{
   int i, j; 
   int sWord;     /* Which word (0 - 25) we're on */

   /* First, get the words loaded up */
   
   for ( i = 0, j = sRow * 5; i < 5; ++i, ++j )
      fill_cell( i, szWords4Card[j] );
   
   for ( i = 0; i < 6; ++i )
   {
      fprintf( fpOut, "  %c%14.14s%c%14.14s%c%14.14s%c%14.14s%c%14.14s%c\n",
               LEFT_SIDE, &szCells[0][ i * 14 ],
               MID_VERT,  &szCells[1][ i * 14 ],
               MID_VERT,  &szCells[2][ i * 14 ],
               MID_VERT,  &szCells[3][ i * 14 ],
               MID_VERT,  &szCells[4][ i * 14 ],
               MID_VERT,  &szCells[5][ i * 14 ],
               RIGHT_SIDE );
   }
}   

/*****************************************************************************/
/** FUNCTION: FILL_CELL()                                                   **/
/**                                                                         **/
/**  PURPOSE: Put a word into a cell                                        **/
/**                                                                         **/
/**    INPUT:                                                               **/
/**           int sCell - Which cell we're filling                          **/
/**           char *szWord - word to insert                                 **/
/**                                                                         **/
/**   OUTPUT: (void)                                                        **/
/**                                                                         **/
/**  HISTORY:                                                               **/
/**                                                                         **/
/**    NOTES:                                                               **/
/**           The cell is filled with the word as neatly centered as        **/
/**             possible.                                                   **/
/**                                                                         **/
/*****************************************************************************/
void fill_cell( int sCell, char *szWord )
{
   int i, j; 
   int sLen;      /* Length of 'word' */
   int sSeg[6];   /* Length of this segment */
   int sNumLines; /* Number of lines we have */
   int sLine;     /* Which line we're on */
   int sOff;      /* Offset into this line */
   int sChar;     /* Offset into the word */
   int sWord;     /* Which word (0 - 25) we're on */

   memset( szCells[sCell], ' ', ( 6 * 14 ) );   /* Blank the cell */
   
   /**********************************************************/
   /**  First, scan through the word to see how it will fit **/
   /**********************************************************/

   sLen = strlen( szWord );
   
      /* Count how many line breaks we have */
   i = 0;
   sNumLines = 0;
      
   while ( i < sLen )
   {
      for ( j = 0; i < sLen && j < 14; ++j )
      {
         if ( szWord[ i++ ] == '|' )
            break;   /* End this line early */
      } /* End for as many chars as will fit */
      
      ++sNumLines;   /* Count the line */
   } /* end while counting lines */

   /* If there are more than 6 lines, just do what we can */
   if ( sNumLines > 6 )
      sNumLines = 6;
   
   /* Now go through and get the length of each segment */
   
   for ( i = 0; i < 6; ++i )
      sSeg[i] = 0;               /* Init to unused */

   sLine = ( 3 - ( sNumLines / 2 ) );  /* Figure out starting line */
   
   for ( i = 0; i < sLen && sLine < 6; )
   {
      for ( j = 0; i < sLen && j < 14; ++j )
      {
         if ( szWord[ i++ ] == '|' )
            break;   /* End this line early */
      } /* End for as many chars as will fit */
      
      sSeg[sLine] = j; /* Note length */
      ++sLine;         /* Next line */
   } /* end for getting all segment lengths */

   /* Now transfer the word to the cell */         
   sChar = 0;
   
   for ( sLine = 0; sLine < 6; ++sLine )
   {
      if ( sSeg[ sLine ] == 0 )
         continue;            /* Nothing goes on this line */
      
      sOff = ( 14 - sSeg[sLine] ) / 2;  /* Starting offset in line */
      sOff += 14 * sLine;               /* Add offset to line */
      
      for ( i = 0; i < sSeg[sLine]; ++i )
      {
         if ( szWord[ sChar ] != '|' )
            szCells[sCell][sOff++] = szWord[sChar];
         ++sChar;
      }

      if ( szWord[ sChar ] == '|' )
         ++sChar; /* Bump in case on a '|' */

   } /* End for each possible line in cell */
}

/*****************************************************************************/
/** FUNCTION:                                                               **/
/**                                                                         **/
/**  PURPOSE:                                                               **/
/**                                                                         **/
/**    INPUT: (void)                                                        **/
/**                                                                         **/
/**   OUTPUT: (void)                                                        **/
/**                                                                         **/
/**  HISTORY:                                                               **/
/**                                                                         **/
/**    NOTES:                                                               **/
/**                                                                         **/
/*****************************************************************************/

